// HeliDecAmps class definition file. -*- C++ -*-
// Copyright 2012 Bertram Kopf

#include <getopt.h>
#include <fstream>
#include <string>
#include <mutex>

#include "PwaUtils/HeliDecAmps.hh"
#include "qft++/relativistic-quantum-mechanics/Utils.hh"
#include "ErrLogger/ErrLogger.hh"
#include "PwaUtils/DataUtils.hh"
#include "PwaUtils/IsobarHeliDecay.hh"
//#include "PwaUtils/XdecAmpRegistry.hh"
#include "Particle/Particle.hh"


HeliDecAmps::HeliDecAmps(boost::shared_ptr<IsobarHeliDecay> theDec) :
  AbsXdecAmp(theDec)
  ,_JPClamlams(theDec->JPClamlamAmps())
  ,_factorMag(1.)
{
  if(_JPClamlams.size()>0) _factorMag=1./sqrt(_JPClamlams.size());
  Particle* daughter1=_decay->daughter1Part();
  Particle* daughter2=_decay->daughter2Part();
  _parityFactor=_JPCPtr->P*daughter1->theParity()*daughter2->theParity()*pow(-1,_JPCPtr->J-daughter1->J()-daughter2->J());
  Info << "_parityFactor=\t" << _parityFactor << endmsg;

  bool identicalDaughters=false;
  if( (*daughter1)==(*daughter2)) identicalDaughters=true;
  
  //fill JPClamlamSymMap
  std::vector< boost::shared_ptr<const JPClamlam> >::iterator it;
  for(it=_JPClamlams.begin(); it!=_JPClamlams.end(); ++it){
    boost::shared_ptr<const JPClamlam> currentSym(new JPClamlam(*it, -(*it)->lam1, -(*it)->lam2, _parityFactor));
    //    std::vector< boost::shared_ptr<const JPClamlam> > currentLPClamlamVec=_JPClamlamSymMap[*it];
    _JPClamlamSymMap[*it].push_back(currentSym);
    if(identicalDaughters && ( fabs((*it)->lam1) != fabs((*it)->lam2) ) ){
    	boost::shared_ptr<const JPClamlam> currentSymIdPart1(new JPClamlam(*it, (*it)->lam2, (*it)->lam1, (*it)->parityFactor));
    	boost::shared_ptr<const JPClamlam> currentSymIdPart2(new JPClamlam(*it, -(*it)->lam2, -(*it)->lam1, _parityFactor));
    	_JPClamlamSymMap[*it].push_back(currentSymIdPart1);
    	_JPClamlamSymMap[*it].push_back(currentSymIdPart2);
    } 
  }
}

HeliDecAmps::HeliDecAmps(boost::shared_ptr<AbsDecay> theDec) :
  AbsXdecAmp(theDec)
{
  Particle* daughter1=_decay->daughter1Part();
  Particle* daughter2=_decay->daughter2Part();
  _parityFactor=_JPCPtr->P*daughter1->theParity()*daughter2->theParity()*pow(-1,_JPCPtr->J-daughter1->J()-daughter2->J()); 
  Info << "_parityFactor=\t" << _parityFactor << endmsg; 
}

HeliDecAmps::~HeliDecAmps()
{
}


complex<double> HeliDecAmps::XdecPartAmp(Spin lamX, Spin lamDec, short fixDaughterNr, EvtData* theData, Spin lamFs, AbsXdecAmp* grandmaAmp){
  complex<double> result(0.,0.);

  bool lamFs_daughter1=false;
  if( _daughter1IsStable && _Jdaughter1>0) lamFs_daughter1=true;
  
  bool lamFs_daughter2=false;
  if( _daughter2IsStable && _Jdaughter2>0) lamFs_daughter2=true;

  std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess >::iterator it;

  for(it=_currentParamMagLamLams.begin(); it!=_currentParamMagLamLams.end(); ++it){
    boost::shared_ptr<const JPClamlam> currentJPClamlam=it->first;
    if( fabs(lamX) > currentJPClamlam->J) continue;

    double theMag=it->second;
    double thePhi=_currentParamPhiLamLams[currentJPClamlam];
    complex<double> expi(cos(thePhi), sin(thePhi));

    Spin lambda1= currentJPClamlam->lam1;
    Spin lambda2= currentJPClamlam->lam2;
    Spin lambda = lambda1-lambda2;
    if( fabs(lambda) > currentJPClamlam->J) continue;
    if(lamFs_daughter1 && lamFs!=lambda1) continue;
    if(lamFs_daughter2 && lamFs!=lambda2) continue;
    if(fixDaughterNr==1 && lamDec!=lambda1) continue;
    if(fixDaughterNr==2 && lamDec!=lambda2) continue;

    complex<double> amp = currentJPClamlam->parityFactor*theMag*expi*conj( theData->WignerDsString[_wignerDKey][currentJPClamlam->J][lamX][lambda]);
    result+=amp;
  }
  //  result*=sqrt((2.*_JPCPtr->J+1.)/12.56637);
  result*=sqrt(2.*_JPCPtr->J+1.);
  return result;
}




complex<double> HeliDecAmps::XdecAmp(Spin lamX, EvtData* theData, Spin lamFs, AbsXdecAmp* grandmaAmp){

  complex<double> result(0.,0.);  

  if( fabs(lamX) > _JPCPtr->J) return result;

  int evtNo=theData->evtNo;
  std::string currentKey=_absDyn->grandMaKey(grandmaAmp);
  
  if ( _cacheAmps && !_recalculate){
    result=_cachedGrandmaAmpMap[currentKey][evtNo][lamX][lamFs];
    return result;
  }

  std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess >::iterator it;

  for(it=_currentParamMagLamLams.begin(); it!=_currentParamMagLamLams.end(); ++it){

    Spin lambda1= it->first->lam1;
    Spin lambda2= it->first->lam2;
    Spin lambda = lambda1-lambda2;
    if( fabs(lambda) > it->first->J) continue;

    if(_enabledlamFsDaughter1 && lamFs!=lambda1) continue;
    if(_enabledlamFsDaughter2 && lamFs!=lambda2) continue;

    double theMag=it->second;
    double thePhi=_currentParamPhiLamLams[it->first];
    complex<double> expi(cos(thePhi), sin(thePhi));
    complex<double> amp = it->first->parityFactor*theMag*expi*conj( theData->WignerDsString[_wignerDKey][it->first->J][lamX][lambda]);
    result+=amp*daughterAmp(lambda1, lambda2, theData, lamFs, this);
  }

  result*=_absDyn->eval(theData, grandmaAmp);
  result*=sqrt(2.*_JPCPtr->J+1.);

  if ( _cacheAmps){
     theMutex.lock();
     _cachedGrandmaAmpMap[currentKey][evtNo][lamX][lamFs]=result;
     theMutex.unlock();
}

  return result;
}

void  HeliDecAmps::getDefaultParams(fitParams& fitVal, fitParams& fitErr){

  std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess > currentMagValMap;
  std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess > currentPhiValMap;
  std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess > currentMagErrMap;
  std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess > currentPhiErrMap;

  std::vector< boost::shared_ptr<const JPClamlam> >::const_iterator itlamlam;
  for(itlamlam=_JPClamlams.begin(); itlamlam!=_JPClamlams.end(); ++itlamlam){
    currentMagValMap[*itlamlam]=_factorMag;
    currentPhiValMap[*itlamlam]=0.;
    currentMagErrMap[*itlamlam]=_factorMag;
    currentPhiErrMap[*itlamlam]=0.3;
  }

  fitVal.MagLamLams[_key]=currentMagValMap;
  fitVal.PhiLamLams[_key]=currentPhiValMap;
  fitErr.MagLamLams[_key]=currentMagErrMap;
  fitErr.PhiLamLams[_key]=currentPhiErrMap;

  _absDyn->getDefaultParams(fitVal, fitErr);

  if(!_daughter1IsStable) _decAmpDaughter1->getDefaultParams(fitVal, fitErr);
  if(!_daughter2IsStable) _decAmpDaughter2->getDefaultParams(fitVal, fitErr);  
}

void HeliDecAmps::print(std::ostream& os) const{
  return; //dummy
}


bool HeliDecAmps::checkRecalculation(fitParams& theParamVal){
  _recalculate=false;

   if(_absDyn->checkRecalculation(theParamVal)) _recalculate=true; 

   if(!_daughter1IsStable) {
     if(_decAmpDaughter1->checkRecalculation(theParamVal)) _recalculate=true;
   }
   if(!_daughter2IsStable){
     if(_decAmpDaughter2->checkRecalculation(theParamVal)) _recalculate=true;
   }

   if(!_recalculate){
     std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess >& magMap=theParamVal.MagLamLams[_key];
     std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess >& phiMap=theParamVal.PhiLamLams[_key];
     std::vector< boost::shared_ptr<const JPClamlam> >::iterator it;
     for (it=_JPClamlams.begin(); it!=_JPClamlams.end(); ++it){
       double theMag=magMap[*it];
       double thePhi=phiMap[*it];
       
       if ( fabs(theMag - _currentParamMagLamLams[*it])  > 1.e-10 ){
	 _recalculate=true;
	 return _recalculate;
       }
       if ( fabs(thePhi - _currentParamPhiLamLams[*it])  > 1.e-10 ){
	 _recalculate=true;
	 return _recalculate;
       }
     }
   }
   return _recalculate;
}
 

void  HeliDecAmps::updateFitParams(fitParams& theParamVal){
   std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess >& magMap=theParamVal.MagLamLams[_key];
   std::map< boost::shared_ptr<const JPClamlam>, double, pawian::Collection::SharedPtrLess >& phiMap=theParamVal.PhiLamLams[_key];

   std::vector< boost::shared_ptr<const JPClamlam> >::iterator it;
   for (it=_JPClamlams.begin(); it!=_JPClamlams.end(); ++it){
     double theMag=magMap[*it];
     double thePhi=phiMap[*it];
     _currentParamMagLamLams[*it]=theMag;
     _currentParamPhiLamLams[*it]=thePhi;

     std::vector< boost::shared_ptr<const JPClamlam> >& currentLPClamlamVec=_JPClamlamSymMap[*it];     
     std::vector< boost::shared_ptr<const JPClamlam> >::iterator itLamLam;
     for (itLamLam=currentLPClamlamVec.begin(); itLamLam!=currentLPClamlamVec.end(); ++itLamLam){
       _currentParamMagLamLams[*itLamLam]=theMag; 
       _currentParamPhiLamLams[*itLamLam]=thePhi;
     }
   }


   _absDyn->updateFitParams(theParamVal);

   if(!_daughter1IsStable) _decAmpDaughter1->updateFitParams(theParamVal);
   if(!_daughter2IsStable) _decAmpDaughter2->updateFitParams(theParamVal);
   
}
