/**
 * @file experimentstate.cpp
 *
 * @author Tobias Triffterer
 *
 * @brief Data Struct to keep Key Parameters of the Experimental Setup
 *
 * Rutherford Experiment Lab Course Online
 * Copyright © 2021 Ruhr-Universität Bochum, Institut für Experimentalphysik I
 * https://www.ep1.ruhr-uni-bochum.de/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **/

#ifndef LIBFP311ONLINE_EXPERIMENTSTATE_H
#define LIBFP311ONLINE_EXPERIMENTSTATE_H

#include <cinttypes>

#include "boundednumeral.h"
#include "clilogger.h"
#include "command.h"
#include "fp311online_export.h"

namespace Fp311Online
{

struct FP311ONLINE_EXPORT ExperimentState final
{
public:
    enum class AdcState : bool
    {
        Running = true,
        Stopped = false
    };
    enum class BeamHoleState : uint8_t
    {
        Closed = 0,
        Open = 1,
        GoldFoil = 2,
    };

    static const constexpr uint16_t NumberOfAdcChannels = 4096;

    using PressureHectoPascal = UInt32BoundedDouble<0, 1500>;
    using PressureTorr = UInt32BoundedDouble<0, 1130>; // 1500 hPa ≈ 1125 Torr, so 1130 Torr as upper limit should make sure that the conversion from hPa to Torr never fails
    using TargetPositionMilliMeter = UInt32BoundedDouble<14, 200>;
    using AdcConversion = GenericBoundedNumeral<uint16_t, uint16_t, 0, NumberOfAdcChannels - 1>;
    using VacuumValveOpening = GenericBoundedNumeral<uint8_t, uint8_t, 0, 10>;

    uint64_t id = 0;
    AdcState adcState = AdcState::Stopped;
    BeamHoleState beamHoleState = BeamHoleState::Closed;
    PressureHectoPascal pressure = PressureHectoPascal::minimum ;
    TargetPositionMilliMeter targetPosition = TargetPositionMilliMeter::minimum;
    AdcConversion adcThreshold = AdcConversion::minimum;
    VacuumValveOpening vacuumValve = VacuumValveOpening::minimum;

    static PressureTorr convertHectoPascalToTorr(const PressureHectoPascal& pressure);

    Protocol::Command getStateCommand() const;

    ExperimentState() = default;
    ExperimentState(const uint64_t experimentId);
    ~ExperimentState() = default;
    ExperimentState ( const ExperimentState& other ) = default;
    ExperimentState ( ExperimentState&& other ) = default;
    ExperimentState& operator= ( const ExperimentState& other ) = default;
    ExperimentState& operator= ( ExperimentState&& other ) = default;

private:
    const CliLogger::ComponentLogger logError = CliLogger::createComponentLogger(QStringLiteral("ExperimentState"), LogLevel::Error);
    const CliLogger::ComponentLogger logWarning = CliLogger::createComponentLogger(QStringLiteral("ExperimentState"), LogLevel::Warning);
    const CliLogger::ComponentLogger logInfo = CliLogger::createComponentLogger(QStringLiteral("ExperimentState"), LogLevel::Info);
    const CliLogger::ComponentLogger logDebug = CliLogger::createComponentLogger(QStringLiteral("ExperimentState"), LogLevel::Debug);
};

}

#endif // LIBFP311ONLINE_EXPERIMENTSTATE_H
