/**
 * @file experimentstate.cpp
 *
 * @author Tobias Triffterer
 *
 * @brief Data Struct to keep Key Parameters of the Experimental Setup
 *
 * Rutherford Experiment Lab Course Online
 * Copyright © 2021 Ruhr-Universität Bochum, Institut für Experimentalphysik I
 * https://www.ep1.ruhr-uni-bochum.de/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **/

#include "experimentstate.h"

using namespace Fp311Online;

ExperimentState::ExperimentState(const uint64_t experimentId)
    : id(experimentId)
{}

ExperimentState::PressureTorr ExperimentState::convertHectoPascalToTorr(const PressureHectoPascal& pressure)
{
    return pressure * 76000.0 / 101325.0;
}

Protocol::Command ExperimentState::getStateCommand() const
{
    const QString adcstate = (adcState == AdcState::Running) ? QStringLiteral("running") : QStringLiteral("stopped");
    const QString beamhole = [this]() -> QString {
        switch (beamHoleState) {
            case BeamHoleState::Closed:
            default:
                return QStringLiteral("closed");
            case BeamHoleState::Open:
                return QStringLiteral("open");
            case BeamHoleState::GoldFoil:
                return QStringLiteral("goldfoil"); }
        }();

    return Protocol::Command(
               Protocol::Action::updateExperimentState,
               Protocol::Command::Arguments {
                    std::make_pair(QStringLiteral("adcstate"), adcstate),
                    std::make_pair(QStringLiteral("beamhole"), beamhole),
                    std::make_pair(QStringLiteral("pressurehPa"), QString::number(pressure)),
                    std::make_pair(QStringLiteral("targetposition"), QString::number(targetPosition)),
                    std::make_pair(QStringLiteral("adcthreshold"), QString::number(adcThreshold)),
                    std::make_pair(QStringLiteral("vacuumvalve"), QString::number(vacuumValve))
               },
               QString() // Token will be added by class that requested the Command object
           );
}
