/**
 * @file rootcanvaswidget.cpp
 *
 * @author Tobias Triffterer
 *
 * @brief Qt Widget to host a ROOT Canvas
 *
 * Rutherford Experiment Lab Course Online
 * Copyright © 2021 Ruhr-Universität Bochum, Institut für Experimentalphysik I
 * https://www.ep1.ruhr-uni-bochum.de/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **/

/**
 * The code in this class has been inspired by the following ROOT forum thread
 * https://root-forum.cern.ch/t/gvirtualx-addwindow-problem/25215/5
 * and this example posted there:
 * https://root-forum.cern.ch/uploads/default/original/2X/4/4f1f5269962b0622a04fa6093c805f5573669df4.gz
 **/

#include <TVirtualX.h>

#include "clilogger.h"
#include "rootcanvaswidget.h"

using namespace Fp311Online;

RootCanvasWidget::RootCanvasWidget(QWidget* const parent)
    : QWidget(parent),
      _canvas(nullptr),
      _ignoreUserInput(false)
{
    // set options needed to properly update the canvas when resizing the widget
    // and to properly handle context menus and mouse move events
    setAttribute(Qt::WA_PaintOnScreen, true);
    setAttribute(Qt::WA_OpaquePaintEvent, true);
    setAttribute(Qt::WA_NativeWindow, true);
    setUpdatesEnabled(false);
    setMouseTracking(true);
    setMinimumSize(300, 200);

    const Int_t rootWinId = gVirtualX->AddWindow(
                                static_cast<ULong_t>(winId()),
                                static_cast<UInt_t>(width()),
                                static_cast<UInt_t>(height())
                            );
    _canvas.reset(
        new TCanvas(
            "RootCanvasWidget",
            width(),
            height(),
            rootWinId
        )
    );
    _canvas->Draw();
    _canvas->Show();
}

void RootCanvasWidget::resizeEvent(QResizeEvent* const event)
{
    QWidget::resizeEvent(event);

    if (_canvas) {
        _canvas->SetCanvasSize(
            static_cast<UInt_t>(event->size().width()),
            static_cast<UInt_t>(event->size().height())
        );
        _canvas->Draw();
        _canvas->Modified();
        _canvas->Resize();
        _canvas->Update();
    }
}

void RootCanvasWidget::paintEvent(QPaintEvent* const event)
{
    QWidget::paintEvent(event);

    if (_canvas) {
        _canvas->Resize();
        _canvas->Update();
    }
}

void RootCanvasWidget::mouseMoveEvent(QMouseEvent* const event)
{
    if (!_ignoreUserInput && _canvas) {
#pragma GCC diagnostic push // Silence compiler warning that not all enum values are handled in
#pragma GCC diagnostic ignored "-Wswitch-enum" // the switch statement, which is perfectly fine here.
#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wswitch-enum"
#endif
        if (event->buttons() & Qt::LeftButton) {
            _canvas->HandleInput(kButton1Motion, event->x(), event->y());
        } else if (event->buttons() & Qt::MiddleButton) {
            _canvas->HandleInput(kButton2Motion, event->x(), event->y());
        } else if (event->buttons() & Qt::RightButton) {
            _canvas->HandleInput(kButton3Motion, event->x(), event->y());
        } else {
            _canvas->HandleInput(kMouseMotion, event->x(), event->y());
        }
#ifdef __clang__
#pragma clang diagnostic pop
#endif
#pragma GCC diagnostic pop
    } else {
        QWidget::mouseMoveEvent(event);
    }
}

void RootCanvasWidget::mousePressEvent(QMouseEvent* const event)
{

    if (!_ignoreUserInput && _canvas) {
#pragma GCC diagnostic push // Silence compiler warning that not all enum values are handled in
#pragma GCC diagnostic ignored "-Wswitch-enum" // the switch statement, which is perfectly fine here.
#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wswitch-enum"
#endif
        switch (event->button()) {
            case Qt::LeftButton:
                _canvas->HandleInput(kButton1Down, event->x(), event->y());
                break;
            case Qt::MiddleButton:
                _canvas->HandleInput(kButton2Down, event->x(), event->y());
                break;
            case Qt::RightButton:
                _canvas->HandleInput(kButton3Down, event->x(), event->y());
                break;
            default:
                break;
        }
#pragma GCC diagnostic pop
#ifdef __clang__
#pragma clang diagnostic pop
#endif
    } else {
        QWidget::mousePressEvent(event);
    }
}

void RootCanvasWidget::mouseReleaseEvent(QMouseEvent* const event)
{
    if (!_ignoreUserInput && _canvas) {
#pragma GCC diagnostic push // Silence compiler warning that not all enum values are handled in
#pragma GCC diagnostic ignored "-Wswitch-enum" // the switch statement, which is perfectly fine here.
#ifdef __clang__
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wswitch-enum"
#endif
        switch (event->button()) {
            case Qt::LeftButton:
                _canvas->HandleInput(kButton1Up, event->x(), event->y());
                break;
            case Qt::MiddleButton:
                _canvas->HandleInput(kButton2Up, event->x(), event->y());
                break;
            case Qt::RightButton:
                _canvas->HandleInput(kButton3Up, event->x(), event->y());
                break;
            default:
                break;
        }
#ifdef __clang__
#pragma clang diagnostic pop
#endif
#pragma GCC diagnostic pop
    } else {
        QWidget::mouseReleaseEvent(event);
    }
}

TCanvas* RootCanvasWidget::getCanvas() const
{
    return _canvas.get();
}

bool RootCanvasWidget::ignoreUserInput() const
{
    return _ignoreUserInput;
}

void RootCanvasWidget::setIgnoreUserInput(const bool ignore)
{
    _ignoreUserInput = ignore;
}
