/**
 * @file boundaryexceededexception.h
 *
 * @author Tobias Triffterer
 *
 * @brief Exception Template for BoundedNumeral out of Range Error
 *
 * Rutherford Experiment Lab Course Online
 * Copyright © 2021 Ruhr-Universität Bochum, Institut für Experimentalphysik I
 * https://www.ep1.ruhr-uni-bochum.de/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **/

#ifndef LIBFP311ONLINE_BOUNDARYEXCEEDEDEXCEPTION_H
#define LIBFP311ONLINE_BOUNDARYEXCEEDEDEXCEPTION_H

#include <sstream>
#include <stdexcept>
#include <string>
#include <type_traits>

#include "fp311online_export.h"

namespace Fp311Online
{

template<typename T, typename L, L min, L max>
class FP311ONLINE_EXPORT BoundaryExceededException final : public std::runtime_error
{
public:
    using valueType = T;
    using boundaryType = L;

    constexpr static valueType minimum = min;
    constexpr static valueType maximum = max;
    constexpr static boundaryType lowerBoundary = min;
    constexpr static boundaryType upperBoundary = max;
    constexpr static bool moveConstructorCanThrow = std::is_nothrow_move_constructible<valueType>::value && std::is_nothrow_move_constructible<std::string>::value;
    constexpr static bool moveAssignmentCanThrow = std::is_nothrow_move_assignable<valueType>::value && std::is_nothrow_move_assignable<std::string>::value;

    BoundaryExceededException ( const valueType& value )
        : std::runtime_error ( createErrorString ( value ) ),
          _value ( value ),
          _message ( createErrorString ( value ) )
    {}

    ~BoundaryExceededException() override = default;
    BoundaryExceededException ( const BoundaryExceededException& other ) = default;
    BoundaryExceededException ( BoundaryExceededException&& other ) noexcept ( moveConstructorCanThrow ) = default;
    BoundaryExceededException& operator= ( const BoundaryExceededException& other ) = default;
    BoundaryExceededException& operator= ( BoundaryExceededException&& other ) noexcept ( moveAssignmentCanThrow ) = default;

    const std::string& getMessage() const noexcept
    {
        return _message;
    }

    const char* what() const noexcept override
    {
        return _message.c_str();
    }
private:
    valueType _value;
    std::string _message;

    static std::string createErrorString ( const valueType& value )
    {
        std::stringstream tmp;
        tmp << "The value " << value << " exceeds the valid range [" << minimum << ";" << maximum << "] for this bounded numeral.";
        return tmp.str();
    }
};

}

#endif // LIBFP311ONLINE_BOUNDARYEXCEEDEDEXCEPTION_H
