/**
 * @file boundednumeral.h
 *
 * @author Tobias Triffterer
 *
 * @brief Template for a Numeral that always stays within a defined Range
 *
 * Rutherford Experiment Lab Course Online
 * Copyright © 2021 Ruhr-Universität Bochum, Institut für Experimentalphysik I
 * https://www.ep1.ruhr-uni-bochum.de/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **/

#ifndef LIBFP311ONLINE_BOUNDEDNUMERAL_H
#define LIBFP311ONLINE_BOUNDEDNUMERAL_H

#include <cinttypes>
#include <stdexcept>

#include "boundaryexceededexception.h"
#include "fp311online_export.h"

namespace Fp311Online
{

template<typename T, typename L, L min, L max>
class FP311ONLINE_EXPORT GenericBoundedNumeral final
{
public:
    using valueType = T;
    using boundaryType = L;
    using BoundaryExceeded = BoundaryExceededException<T, L, min, max>;

    static constexpr valueType minimum = min;
    static constexpr valueType maximum = max;
    static constexpr boundaryType lowerBoundary = min;
    static constexpr boundaryType upperBoundary = max;

    constexpr GenericBoundedNumeral() noexcept : value ( minimum ) {}
    ~GenericBoundedNumeral() noexcept = default;
    constexpr GenericBoundedNumeral ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept = default;
    constexpr GenericBoundedNumeral ( GenericBoundedNumeral<T, L, min, max>&& other ) noexcept = default;
    constexpr GenericBoundedNumeral<T, L, min, max>& operator = ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept = default;
    constexpr GenericBoundedNumeral<T, L, min, max>& operator = ( GenericBoundedNumeral<T, L, min, max>&& other ) noexcept = default;

    constexpr GenericBoundedNumeral ( const valueType& source ) noexcept ( false )
        : value ( checkValue ( source ) )
    {}

    constexpr operator valueType() const noexcept
    {
        return value;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator = ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( other );
        return *this;
    }

    constexpr bool operator == ( const GenericBoundedNumeral<T, L, min, max>& other ) const noexcept
    {
        return value == other.value;
    }

    constexpr bool operator != ( const GenericBoundedNumeral<T, L, min, max>& other ) const noexcept
    {
        return value != other.value;
    }

    constexpr bool operator < ( const GenericBoundedNumeral<T, L, min, max>& other ) const noexcept
    {
        return value < other.value;
    }

    constexpr bool operator < ( const valueType& other ) const noexcept
    {
        return value < other;
    }

    constexpr bool operator > ( const GenericBoundedNumeral<T, L, min, max>& other ) const noexcept
    {
        return value > other.value;
    }

    constexpr bool operator > ( const valueType& other ) const noexcept
    {
        return value > other;
    }

    constexpr bool operator <= ( const GenericBoundedNumeral<T, L, min, max>& other ) const noexcept
    {
        return value <= other.value;
    }

    constexpr bool operator <= ( const valueType& other ) const noexcept
    {
        return value <= other;
    }

    constexpr bool operator >= ( const GenericBoundedNumeral<T, L, min, max>& other ) const noexcept
    {
        return value >= other.value;
    }

    constexpr bool operator >= ( const valueType& other ) const noexcept
    {
        return value >= other;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator += ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept ( false )
    {
        value = checkValue ( value + other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator += ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( value + other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator -= ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept ( false )
    {
        value = checkValue ( value - other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator -= ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( value - other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator *= ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept ( false )
    {
        value = checkValue ( value * other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator *= ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( value * other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator /= ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept ( false )
    {
        value = checkValue ( value / other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator /= ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( value / other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator %= ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept ( false )
    {
        value = checkValue ( value % other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator %= ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( value % other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator &= ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept ( false )
    {
        value = checkValue ( value & other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator &= ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( value & other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator |= ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept ( false )
    {
        value = checkValue ( value | other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator |= ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( value | other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator ^= ( const GenericBoundedNumeral<T, L, min, max>& other ) noexcept ( false )
    {
        value = checkValue ( value ^ other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator ^= ( const valueType& other ) noexcept ( false )
    {
        value = checkValue ( value ^ other );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator++() noexcept ( false )
    {
        value = checkValue ( ++value );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max>& operator--() noexcept ( false )
    {
        value = checkValue ( --value );
        return *this;
    }

    constexpr GenericBoundedNumeral<T, L, min, max> operator++ ( int ) noexcept ( false )
    {
        const T tmp = value;
        value = checkValue ( ++value );
        return tmp;
    }

    constexpr GenericBoundedNumeral<T, L, min, max> operator-- ( int ) noexcept ( false )
    {
        const T tmp = value;
        value = checkValue ( --value );
        return tmp;
    }

    constexpr valueType get() const noexcept
    {
        return value;
    }

    constexpr inline void set ( const valueType& newvalue ) noexcept ( false )
    {
        value = checkValue ( newvalue );
    }

    constexpr inline void setOrAdjust ( const valueType& newvalue ) noexcept
    {
        value = newvalue < minimum ? minimum : ( newvalue > maximum ? maximum : newvalue );
    }

    constexpr static bool checkValueIsInRange ( const valueType& value ) noexcept
    {
        return value >= minimum && value <= maximum;
    }

    constexpr static GenericBoundedNumeral<T, L, min, max> createAndAdjust ( const valueType& source ) noexcept
    {
        return GenericBoundedNumeral<T, L, min, max> ( source < minimum ? minimum : ( source > maximum ? maximum : source ) );
    }

private:
    valueType value;

    constexpr static const valueType& checkValue ( const valueType& source ) noexcept ( false )
    {
        if ( source < minimum || source > maximum )
            throw BoundaryExceeded ( source );
        return source;
    }

};

template<typename T, T min, T max> using BoundedNumeral = GenericBoundedNumeral<T, T, min, max> ;

template<int64_t min, int64_t max> using Int64BoundedFloat = GenericBoundedNumeral<float, int64_t, min, max>;

template<int64_t min, int64_t max> using Int64BoundedDouble = GenericBoundedNumeral<double, int64_t, min, max>;

template<uint64_t min, uint64_t max> using UInt64BoundedFloat = GenericBoundedNumeral<float, uint64_t, min, max>;

template<uint64_t min, uint64_t max> using UInt64BoundedDouble = GenericBoundedNumeral<double, uint64_t, min, max>;

template<int32_t min, int32_t max> using Int32BoundedFloat = GenericBoundedNumeral<float, int32_t, min, max>;

template<int32_t min, int32_t max> using Int32BoundedDouble = GenericBoundedNumeral<double, int32_t, min, max>;

template<uint32_t min, uint32_t max> using UInt32BoundedFloat = GenericBoundedNumeral<float, uint32_t, min, max>;

template<uint32_t min, uint32_t max> using UInt32BoundedDouble = GenericBoundedNumeral<double, uint32_t, min, max>;

}

#endif // LIBFP311ONLINE_BOUNDEDNUMERAL_H

