/**
 * @file command.cpp
 *
 * @author Tobias Triffterer
 *
 * @brief Struct handling Protocol Commands
 *
 * Rutherford Experiment Lab Course Online
 * Copyright © 2021 Ruhr-Universität Bochum, Institut für Experimentalphysik I
 * https://www.ep1.ruhr-uni-bochum.de/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **/

#include <utility>

#include <QJsonDocument>

#include "command.h"

using namespace Fp311Online;
using namespace Fp311Online::Protocol;

static CliLogger::ComponentLogger logError = CliLogger::createComponentLogger(QStringLiteral("Protocol::Command"), LogLevel::Error);
static CliLogger::ComponentLogger logWarning = CliLogger::createComponentLogger(QStringLiteral("Protocol::Command"), LogLevel::Warning);
static CliLogger::ComponentLogger logInfo = CliLogger::createComponentLogger(QStringLiteral("Protocol::Command"), LogLevel::Info);
static CliLogger::ComponentLogger logDebug = CliLogger::createComponentLogger(QStringLiteral("Protocol::Command"), LogLevel::Debug);

Command Command::fromString(const QString& source)
{
    QJsonDocument jsondoc = QJsonDocument::fromJson(source.toUtf8());
    if (!jsondoc.isObject()) {
        logError(QStringLiteral("Received JSON document that is not an object."));
        return Command();
    }
    QJsonObject rootobj = jsondoc.object();
    if (!rootobj.keys().contains(QStringLiteral("action"))) {
        logError(QStringLiteral("Received JSON object does not contain \"action\" property."));
        return Command();
    }

    if (!rootobj.value(QStringLiteral("action")).isString()) {
        logError(QStringLiteral("Action in JSON object is not a string."));
        return Command();
    }

    const Action action = parseActionString(rootobj.value(QStringLiteral("action")).toString());

    const QString token = (rootobj.keys().contains(QStringLiteral("token")) && rootobj.value(QStringLiteral("token")).isString())
                          ? rootobj.value(QStringLiteral("token")).toString()
                          : QString();


    if (token.isEmpty() && action != Action::queryServerInformation && action != Action::authenticate && action != Action::error) {
        logError("JSON object does not contain token although it is required for requested action.");
        return Command();
    }

    switch (action) {
        case Action::invalid:
            return Command();
        case Action::authenticate:
            return parseAuthenticate(rootobj);
        case Action::storeToken:
            return parseStoreToken(rootobj);
        case Action::updateExperimentState:
            return parseUpdateExperimentState(rootobj);
        case Action::setTargetState:
            return parseSetTargetState(rootobj);
        case Action::sendHistogramUpdate:
        case Action::moveTargetCloser:
        case Action::moveTargetFarther:
        case Action::openVacuumValve:
        case Action::closeVacuumValve:
        case Action::startAdc:
        case Action::stopAdc:
        case Action::clearHistogram:
            return Command(action, Arguments(), token);
        case Action::error:
            return parseError(rootobj);
        // Not yet implemented:
        case Action::queryServerInformation:
        case Action::updateHistogram:
        case Action::fillHistogram:
            return Command();
        default:
            return Command();
    }
}

Action Command::parseActionString(const QString& string)
{
    if (string == QStringLiteral("queryServerInformation"))
        return Action::queryServerInformation;
    if (string == QStringLiteral("authenticate"))
        return Action::authenticate;
    if (string == QStringLiteral("storeToken"))
        return Action::storeToken;
    if (string == QStringLiteral("updateExperimentState"))
        return Action::updateExperimentState;
    if (string == QStringLiteral("sendHistogramUpdate"))
        return Action::sendHistogramUpdate;
    if (string == QStringLiteral("updateHistogram"))
        return Action::updateHistogram;
    if (string == QStringLiteral("moveTargetCloser"))
        return Action::moveTargetCloser;
    if (string == QStringLiteral("moveTargetFarther"))
        return Action::moveTargetFarther;
    if (string == QStringLiteral("setTargetState"))
        return Action::setTargetState;
    if (string == QStringLiteral("openVacuumValve"))
        return Action::openVacuumValve;
    if (string == QStringLiteral("closeVacuumValve"))
        return Action::closeVacuumValve;
    if (string == QStringLiteral("startAdc"))
        return Action::startAdc;
    if (string == QStringLiteral("stopAdc"))
        return Action::stopAdc;
    if (string == QStringLiteral("fillHistogram"))
        return Action::fillHistogram;
    if (string == QStringLiteral("clearHistogram"))
        return Action::clearHistogram;
    if (string == QStringLiteral("error"))
        return Action::error;

    logError(QStringLiteral("Action \"") + string + QStringLiteral(" unknown."));
    return Action::invalid;
}

Command Command::parseAuthenticate(const QJsonObject& input)
{
    if (!input.keys().contains(QStringLiteral("startpin")) || !input.value(QStringLiteral("startpin")).isString()) {
        logError(QStringLiteral("Authenticate command does not contain startpin."));
        return Command();
    }

    return Command(
               Action::authenticate,
               Arguments{std::make_pair(QStringLiteral("startpin"), input.value(QStringLiteral("startpin")).toString())},
               QString()
           );
}

Command Command::parseStoreToken(const QJsonObject& input)
{
    if (!input.keys().contains(QStringLiteral("token")) || !input.value(QStringLiteral("token")).isString()
            || !input.keys().contains(QStringLiteral("name")) || !input.value(QStringLiteral("name")).isString()) {
        logError(QStringLiteral("storeToken command does not contain token and/or name."));
        return Command();
    }

    return Command(
               Action::storeToken,
               Arguments{std::make_pair(QStringLiteral("name"), input.value(QStringLiteral("name")).toString())},
               input.value(QStringLiteral("token")).toString()
           );
}

Protocol::Command Protocol::Command::parseError(const QJsonObject& input)
{
    if (!input.keys().contains(QStringLiteral("message")) || !input.value(QStringLiteral("message")).isString()) {
        logError(QStringLiteral("Error command does not contain message."));
        return Command();
    }
    const QString token = (input.keys().contains(QStringLiteral("token")) && input.value(QStringLiteral("token")).isString())
                          ? input.value(QStringLiteral("token")).toString()
                          : QString();

    return Command(
               Action::error,
               Arguments{std::make_pair(QStringLiteral("message"), input.value(QStringLiteral("message")).toString())},
               token
           );
}

Protocol::Command Protocol::Command::parseUpdateExperimentState(const QJsonObject& input)
{
    if (
        !input.keys().contains(QStringLiteral("action")) || !input.value(QStringLiteral("action")).isString()
        || !input.keys().contains(QStringLiteral("adcstate")) || !input.value(QStringLiteral("adcstate")).isString()
        || !input.keys().contains(QStringLiteral("adcthreshold")) || !input.value(QStringLiteral("adcthreshold")).isString()
        || !input.keys().contains(QStringLiteral("beamhole")) || !input.value(QStringLiteral("beamhole")).isString()
        || !input.keys().contains(QStringLiteral("pressurehPa")) || !input.value(QStringLiteral("pressurehPa")).isString()
        || !input.keys().contains(QStringLiteral("targetposition")) || !input.value(QStringLiteral("targetposition")).isString()
        || !input.keys().contains(QStringLiteral("token")) || !input.value(QStringLiteral("token")).isString()
        || !input.keys().contains(QStringLiteral("vacuumvalve")) || !input.value(QStringLiteral("vacuumvalve")).isString()
    ) {
        logError(QStringLiteral("At least one property of the updateExperimentState command is missing or not a string."));
        return Command();
    }

    return Command(
               Action::updateExperimentState,
               Arguments{std::make_pair(QStringLiteral("adcstate"), input.value(QStringLiteral("adcstate")).toString()), std::make_pair(QStringLiteral("adcthreshold"), input.value(QStringLiteral("adcthreshold")).toString()), std::make_pair(QStringLiteral("beamhole"), input.value(QStringLiteral("beamhole")).toString()), std::make_pair(QStringLiteral("pressurehPa"), input.value(QStringLiteral("pressurehPa")).toString()), std::make_pair(QStringLiteral("targetposition"), input.value(QStringLiteral("targetposition")).toString()), std::make_pair(QStringLiteral("vacuumvalve"), input.value(QStringLiteral("vacuumvalve")).toString())},
               input.value(QStringLiteral("token")).toString()
           );
}

Protocol::Command Protocol::Command::parseSetTargetState(const QJsonObject& input)
{
    if (!input.keys().contains(QStringLiteral("newstate")) || !input.value(QStringLiteral("newstate")).isString()) {
        logError(QStringLiteral("setTargetState command does not contain newstate."));
        return Command();
    }
    if (input.value(QStringLiteral("newstate")).toString() != QStringLiteral("open")
            && input.value(QStringLiteral("newstate")).toString() != QStringLiteral("closed")
            && input.value(QStringLiteral("newstate")).toString() != QStringLiteral("goldfoil")) {
        logError(QStringLiteral("newstate parameter of setTargetState command is invalid."));
        return Command();
    }

    return Command(
               Action::setTargetState,
               Arguments{std::make_pair(QStringLiteral("newstate"), input.value(QStringLiteral("newstate")).toString())},
               input.value(QStringLiteral("token")).toString()
           );
}

QString Command::toString() const
{
    if (action == Action::invalid)
        return QString();

    QJsonObject rootobj;
    rootobj.insert(
        QStringLiteral("action"),
        QJsonValue(getActionString(action))
    );

    if (!token.isEmpty())
        rootobj.insert(
            QStringLiteral("token"),
            QJsonValue(token)
        );

    if (!arguments.isEmpty()) {
        for (auto i = arguments.begin(); i != arguments.end(); i++) {
            rootobj.insert(
                i.key(),
                i.value()
            );
        }
    }

    QJsonDocument jsondoc(rootobj);
    return QString::fromUtf8(jsondoc.toJson());
}

QString Command::getActionString(const Action action)
{
    switch (action) {
        case Action::queryServerInformation:
            return QStringLiteral("queryServerInformation");
        case Action::authenticate:
            return QStringLiteral("authenticate");
        case Action::storeToken:
            return QStringLiteral("storeToken");
        case Action::updateExperimentState:
            return QStringLiteral("updateExperimentState");
        case Action::sendHistogramUpdate:
            return QStringLiteral("sendHistogramUpdate");
        case Action::updateHistogram:
            return QStringLiteral("updateHistogram");
        case Action::moveTargetCloser:
            return QStringLiteral("moveTargetCloser");
        case Action::moveTargetFarther:
            return QStringLiteral("moveTargetFarther");
        case Action::setTargetState:
            return QStringLiteral("setTargetState");
        case Action::openVacuumValve:
            return QStringLiteral("openVacuumValve");
        case Action::closeVacuumValve:
            return QStringLiteral("closeVacuumValve");
        case Action::startAdc:
            return QStringLiteral("startAdc");
        case Action::stopAdc:
            return QStringLiteral("stopAdc");
        case Action::fillHistogram:
            return QStringLiteral("fillHistogram");
        case Action::clearHistogram:
            return QStringLiteral("clearHistogram");
        case Action::error:
            return QStringLiteral("error");
        case Action::invalid:
        default:
            return QString();
    }
}
