/**
 * @file clientgui.h
 *
 * @author Tobias Triffterer
 *
 * @brief Client Login Window
 *
 * Rutherford Experiment Lab Course Online
 * Copyright © 2021 Ruhr-Universität Bochum, Institut für Experimentalphysik I
 * https://www.ep1.ruhr-uni-bochum.de/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **/

#ifndef CLIENTGUI_H
#define CLIENTGUI_H

#include <memory>

#include <QMainWindow>
#include <QString>
#include <QShowEvent>
#include <QTimer>
#include <QWebSocket>

#include <RtypesCore.h>
#include <TCanvas.h>
#include <TH1.h>
#include <TROOT.h>
#include <TStyle.h>
#include <TVirtualGL.h>
#include <TVirtualX.h>

#include "action.h"
#include "clilogger.h"
#include "command.h"
#include "experimentstate.h"
#include "rootcanvaswidget.h"
#include "ui_clientgui.h"

namespace Fp311Online
{

class ClientGui : public QMainWindow
{
    Q_OBJECT

public:
    ClientGui(const QString& name, const QString& token, std::unique_ptr<QWebSocket>&& socket);
    ~ClientGui() = default;
    ClientGui(const ClientGui& other) = delete;
    ClientGui(ClientGui&& other) = delete;
    ClientGui& operator=(const ClientGui& other) = delete;
    ClientGui& operator=(ClientGui&& other) = delete;

private:
    static const constexpr double PositionZeroRevolutions = 14;
    static const constexpr double TargetMovementMillimeterPerRevolution = 2;

    const QString _name;
    const QString _token;
    std::unique_ptr<QWebSocket> _socket;
    RootCanvasWidget _rootcanvas;
    std::unique_ptr<TH1F> _histo;
    ExperimentState _state {};
    std::unique_ptr<TCanvas> _canvas {nullptr};
    Ui::ClientGui _ui {};
    QTimer _rootTimer {};

    void handleMessageFromServer(const QString& message);
    void handleSocketError(QAbstractSocket::SocketError error);
    void handleSocketDisconnected();
    void sendSimpleActiontoServer(const Protocol::Action action);
    void setBeamholeState(const ExperimentState::BeamHoleState state);
    static QString getBeamholeStateString(const ExperimentState::BeamHoleState state);

    void displayExperimentState();
    void fillEventsIntoHistogram(const QString& adcchannels);
    void clearHistogram(const bool fromServer = false);

    void saveRootFile();

    void showEvent(QShowEvent* event) override;

    const CliLogger::ComponentLogger logError = CliLogger::createComponentLogger(QStringLiteral("ClientGui"), LogLevel::Error);
    const CliLogger::ComponentLogger logWarning = CliLogger::createComponentLogger(QStringLiteral("ClientGui"), LogLevel::Warning);
    const CliLogger::ComponentLogger logInfo = CliLogger::createComponentLogger(QStringLiteral("ClientGui"), LogLevel::Info);
    const CliLogger::ComponentLogger logDebug = CliLogger::createComponentLogger(QStringLiteral("ClientGui"), LogLevel::Debug);
};

}

#endif // CLIENTGUI_H
