/**
 * @file logingui.cpp
 *
 * @author Tobias Triffterer
 *
 * @brief Client Login Window
 *
 * Rutherford Experiment Lab Course Online
 * Copyright © 2021 Ruhr-Universität Bochum, Institut für Experimentalphysik I
 * https://www.ep1.ruhr-uni-bochum.de/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **/

#include <QApplication>
#include <QColor>
#include <QFontMetrics>
#include <QTimer>
#include <QPalette>
#include <QPixmap>
#include <QSslConfiguration>
#include <QUrl>

#include "cacertificates.h"
#include "clientgui.h"
#include "command.h"
#include "legalstuff.h"
#include "logingui.h"

using namespace Fp311Online;

LoginGui::LoginGui()
{
    _ui.setupUi(this);
    _ui.txtStartPin->setFocus();

    connect(_ui.cmdLogin, &QPushButton::clicked, this, &LoginGui::doLogin);

    if (QApplication::palette().color(QPalette::Window).toHsl().lightness() < LightnessThresholdForWhiteLogo)
        _ui.lblLogoRuhrUni->setPixmap(QPixmap(QStringLiteral(":/logo/logo-ruhr-uni-white.png")));

    const QRect buttonTextSize = _ui.cmdShowLegalStuff->fontMetrics().boundingRect(_ui.cmdShowLegalStuff->text());
    _ui.cmdShowLegalStuff->setMaximumSize(buttonTextSize.width() + 50, _ui.cmdLogin->height() + 5);
    connect(_ui.cmdShowLegalStuff, &QPushButton::clicked, this, &LoginGui::showLegalStuff);
}

void LoginGui::doLogin()
{
    if (_ui.txtUrl->text().isEmpty() || _ui.txtStartPin->text().isEmpty())
        return;

    _ui.txtUrl->setEnabled(false);
    _ui.txtStartPin->setEnabled(false);
    _ui.cmdLogin->setEnabled(false);

    logInfo(QStringLiteral("Opening web socket to address wss://") + _ui.txtUrl->text());
    _socket.reset(new QWebSocket);

    QSslConfiguration sslconfig = _socket->sslConfiguration();
    sslconfig.setCaCertificates(CaCertificates::getCaCertificatesFromResource());
    _socket->setSslConfiguration(sslconfig);

    connect(_socket.get(), QOverload<QAbstractSocket::SocketError>::of(&QWebSocket::error), this, &LoginGui::socketError);
    connect(_socket.get(), &QWebSocket::connected, this, &LoginGui::sendAuthenticateMessage);
    connect(_socket.get(), &QWebSocket::textFrameReceived, this, &LoginGui::receivedMessageFromServer);

    _ui.lblStatus->setText(tr("Connecting to server..."));
    _ui.pbStatus->setMinimum(0); // Activate busy indicator
    _ui.pbStatus->setMaximum(0);

    _socket->open(QUrl(QStringLiteral("wss://") + _ui.txtUrl->text()));
}

void LoginGui::sendAuthenticateMessage()
{
    logInfo(QStringLiteral("Web socket connection established, sending Start PIN ") + _ui.txtStartPin->text());
    _ui.lblStatus->setText(tr("Sending Start PIN to server..."));
    Protocol::Command authenticateCommand(
        Protocol::Action::authenticate,
        Protocol::Command::Arguments { std::make_pair(QStringLiteral("startpin"), _ui.txtStartPin->text()) },
        QString()
    );
    _socket->sendTextMessage(authenticateCommand.toString());
}

void LoginGui::receivedMessageFromServer(const QString& message)
{
    logWarning(QStringLiteral("Received: ") + message);
    const Protocol::Command serverCommand = Protocol::Command::fromString(message);
    if (serverCommand.action == Protocol::Action::invalid)
        return;
    if (serverCommand.action == Protocol::Action::storeToken) {
        logInfo(QStringLiteral("Authentication finished – received token ") + serverCommand.token + QStringLiteral(" from server for user ") + serverCommand.arguments[QStringLiteral("name")]);
        _ui.lblStatus->setText(tr("<html><body><p>Login to server successful!</p><h3>Welcome, ") + serverCommand.arguments[QStringLiteral("name")].toHtmlEscaped() + QStringLiteral("!</h3></body></html>"));
        _ui.pbStatus->setMinimum(0); // Deactivate busy indicator and set progress bar to complete
        _ui.pbStatus->setMaximum(100);
        _ui.pbStatus->setValue(100);

        QTimer::singleShot(
            std::chrono::seconds(3),
            std::bind(&LoginGui::openClientGui, this, serverCommand.arguments[QStringLiteral("name")], serverCommand.token)
        );
    }
}

void LoginGui::openClientGui(const QString name, const QString token)
{
    disconnect(_socket.get(), QOverload<QAbstractSocket::SocketError>::of(&QWebSocket::error), this, &LoginGui::socketError);
    disconnect(_socket.get(), &QWebSocket::connected, this, &LoginGui::sendAuthenticateMessage);
    disconnect(_socket.get(), &QWebSocket::textFrameReceived, this, &LoginGui::receivedMessageFromServer);
    ClientGui* clientgui = new ClientGui(name, token, std::move(_socket));
    clientgui->show();

    hide();
}

void LoginGui::socketError(QAbstractSocket::SocketError error)
{
    _ui.pbStatus->setMinimum(0); // Deactivate busy indicator
    _ui.pbStatus->setMaximum(100);
    _ui.lblStatus->setText(tr("An error occured: ") + _socket->errorString());
    logError(QStringLiteral("Socket Error ") + QString::number(error) + ": " + _socket->errorString());
}

void LoginGui::showLegalStuff()
{
    LegalStuff* ls = new LegalStuff(this);
    ls->setModal(true);
    ls->show();
}
